/*
 * definitions for Fabric Management Agent
 */

#include <stdint.h>
#include <stdio.h>
#include <errno.h>

#define FMA_MAX_ERROR_DEPTH 10
#define FMA_MAX_ERROR_LEN 256

#define LF_ERROR(p)                                 \
  do {                                              \
    errno = 0;					    \
    LF_ERROR_E(p);				    \
  } while (0)

#define LF_ERROR_E(S) do {					\
  char *__errstr;						\
  __errstr = lf_error_msg_fmt S;				\
  fma_save_error(__FILE__, __LINE__, errno, __errstr);		\
  lf_error_msg_free(__errstr);					\
  errno = 0;							\
  goto except;							\
} while (0)

enum fma_xbar_types {
  FMA_XT_ANY,		/* no constraints on xbar types */
  FMA_XT_ID_ONLY,	/* fabric has only xbars with IDs */
  FMA_XT_NO_IDS		/* fabric has no xbars with IDs */
};

enum fma_run_state {
  FMA_RUN_BOOTING,
  FMA_RUN_STANDALONE,
  FMA_RUN_FMS_DIRECT,
  FMA_RUN_PROXY_PENDING,	/* waiting for proxy client ID */
  FMA_RUN_FMS_VIA_PROXY,
  FMA_RUN_COUNT
};

/*
 * structure for all global data 
 */
struct fma_vars {

  int done;	/* set to 1 when time to exit */

  /* settings */
  int debug;
  enum fma_xbar_types xbar_types;
  int map_level;
  int fms_required;		/* no standalone mode if TRUE */
  int my_fma_flags;		/* run flags */
  
  int fma_background;		/* If true, run as daemon */
  char *fma_run;		/* base dir for fma stuff */
  char *fma_logfilename;	/* Name of output log file */
  FILE *fma_logfile;		/* output logfile */

  char hostname[LF_STRING_LEN];	/* local hostname */
  struct lf_host *my_host;	/* host struct in map */

  struct fma_myri *myri;	/* Myricom network info */

  struct fma_fms *fms;		/* FMS related vars */

  struct fma_proxy *proxy;	/* information about proxy when in proxy mode */

  struct fma_settings *settings;	/* settable paramaters */

  struct fma_map_info *map_info;	/* map info */

  struct lf_fabric *fabric;	/* current fabric */

  enum fma_run_state run_state;
  struct fma_standalone_data *stand;	/* standalone info */

  struct fma_tunnel *tunnel;	/* message tunnel data */

  /* Error handling messages */
  int error_index;
  char error_str[FMA_MAX_ERROR_DEPTH][FMA_MAX_ERROR_LEN+1];
  char *error_file[FMA_MAX_ERROR_DEPTH];
  int error_line[FMA_MAX_ERROR_DEPTH];
  int error_errno[FMA_MAX_ERROR_DEPTH];
};

/*
 * function prototypes
 */
void fma_perror(void);
void fma_save_error(char *, int, int, char *);
void fma_reset_error(void);
void fma_exit(int);
void fma_perror_exit(int);
void fma_init_standalone_vars(void);
void fma_init_standalone(void);
void fma_init_map_info_vars(void);
int fma_init_myri_vars(void);
void fma_init_tunnel_vars(void);
int fma_init_myri(void);
int fma_send_myri_info(void);
void fma_init_settings_vars(void);
void fma_free(void *);
void fma_log(const char *fmt, ...);
void fma_update_nic_reply_info(void);


/*
 * external definitions
 */

extern int errno;
extern struct fma_vars A;

/*
 * inline functions
 */
static inline void
fma_reset_errors()
{
  A.error_index = 0;
}

static inline void
fma_set_flag_bits(
  int bits)
{
  A.my_fma_flags |= bits;
  fma_update_nic_reply_info();
}

static inline void
fma_reset_flag_bits(
  int bits)
{
  A.my_fma_flags &= ~bits;
  fma_update_nic_reply_info();
}

/*
 * some constants
 */

#define FMA_IFC_ROUTE_LEN 8	/* longest nic-nic route */
